import * as Cesium from 'cesium'
// import * as satellite from 'satellite.js'

/**
 * 随机生成卫星轨道数据（不依赖 TLE）
 * @param numPoints 采样点数量（轨道上取多少个点）
 * @param timeStep 每两个采样点之间的时间间隔（秒）
 */
export function computeRandomOrbit(
  numPoints = 720,
  timeStep = 6
): {
  positionProperty: Cesium.SampledPositionProperty
  orbitPositions: Cesium.Cartesian3[]
  startTime: Cesium.JulianDate
  stopTime: Cesium.JulianDate
} {
  // 起始时间：以当前时间为起点
  const startTime = Cesium.JulianDate.now()
  const stopTime = Cesium.JulianDate.addSeconds(
    startTime,
    numPoints * timeStep,
    new Cesium.JulianDate()
  )

  const positionProperty = new Cesium.SampledPositionProperty()
  const orbitPositions: Cesium.Cartesian3[] = []

  // ========== 随机轨道参数 ==========
  const earthRadius = 6371e3 // 地球半径 m
  const altitude = 500e3 + Math.random() * 1500e3 // 500~2000 km 高度
  const orbitRadius = earthRadius + altitude

  const inclination = Math.random() * 180 // 倾角 (0~180度)
  const raan = Math.random() * 360 // 升交点赤经 (0~360度)

  for (let i = 0; i < numPoints; i++) {
    const currentTime = Cesium.JulianDate.addSeconds(
      startTime,
      i * timeStep,
      new Cesium.JulianDate()
    )

    // 沿轨道运动的角度（相位）
    const angle = (i / numPoints) * 2 * Math.PI

    // 初始轨道在赤道平面上的位置 (X, Y)
    const x = orbitRadius * Math.cos(angle)
    const y = orbitRadius * Math.sin(angle)
    const z = 0

    let pos = new Cesium.Cartesian3(x, y, z)

    // ========== 旋转到随机倾角和 RAAN ==========
    const rotInc = Cesium.Matrix3.fromRotationX(Cesium.Math.toRadians(inclination))
    const rotRaan = Cesium.Matrix3.fromRotationZ(Cesium.Math.toRadians(raan))
    const rot = Cesium.Matrix3.multiply(rotRaan, rotInc, new Cesium.Matrix3())

    pos = Cesium.Matrix3.multiplyByVector(rot, pos, new Cesium.Cartesian3())

    // 保存轨道点
    orbitPositions.push(pos)
    positionProperty.addSample(currentTime, pos)
  }

  return { positionProperty, orbitPositions, startTime, stopTime }
}

export const satelliteList = [
  {
    name: 'FENGYUN 3E',
    tle1: '1 49008U 21062A   25110.53320445  .00000056  00000-0  47020-4 0  9995',
    tle2: '2 49008  98.7432 113.8688 0001070 179.9589 180.1588 14.19859554196601'
  },
  {
    name: 'HAIYANG 2D',
    tle1: '1 48621U 21043A   25110.51258281  .00000013  00000-0  85647-4 0  9990',
    tle2: '2 48621  65.9966 277.0130 0008040 261.0819  98.9340 13.83372903198126'
  },
  {
    name: 'LANDSAT 8',
    tle1: '1 39084U 13008A   25110.50130981  .00000736  00000-0  17327-3 0  9991',
    tle2: '2 39084  98.1890 181.7768 0001114  99.4629 260.6696 14.57133819648167'
  },
  {
    name: 'NPP',
    tle1: '1 37849U 11061A   25110.49703421  .00000187  00000-0  10942-3 0  9994',
    tle2: '2 37849  98.7546  50.3268 0002538  92.2298 267.9169 14.19521446698466'
  },
  {
    name: 'AQUA',
    tle1: '1 27424U 02022A   25110.49190655  .00001925  00000-0  40090-3 0  9996',
    tle2: '2 27424  98.3731  68.1378 0001092 108.6371   1.6953 14.61156736221563'
  },
  {
    name: 'METOP-B',
    tle1: '1 38771U 12049A   25110.49181281  .00000226  00000-0  12288-3 0  9999',
    tle2: '2 38771  98.5819 166.1057 0001889 116.2950 243.8422 14.21481267653201'
  },
  {
    name: 'FENGYUN 3D',
    tle1: '1 43010U 17072A   25110.49054369  .00000220  00000-0  12607-3 0  9997',
    tle2: '2 43010  98.9580  73.7176 0001801 140.6894 219.4412 14.19611249385010'
  },
  {
    name: 'TERRA',
    tle1: '1 25994U 99068A   25110.48571943  .00000910  00000-0  19615-3 0  9996',
    tle2: '2 25994  98.0028 170.3979 0003633 112.5667 305.9700 14.60635640348012'
  },
  {
    name: 'HAIYANG 2B',
    tle1: '1 43655U 18081A   25110.47966778  .00000017  00000-0  59327-4 0  9992',
    tle2: '2 43655  99.3354 120.5116 0000942  17.4675 342.6509 13.79304456326652'
  },
  {
    name: 'HAIYANG 1D',
    tle1: '1 45721U 20036A   25110.46824375  .00000386  00000-0  15392-3 0  9998',
    tle2: '2 45721  98.6261  55.5181 0000373 328.6016  31.5146 14.34080572254383'
  },
  {
    name: 'FENGYUN 3F',
    tle1: '1 57490U 23111A   25110.46159014  .00000157  00000-0  94462-4 0  9994',
    tle2: '2 57490  98.7292 182.1999 0000353  80.1805 279.9412 14.20029968 88872'
  },
  {
    name: 'HAIYANG 2C',
    tle1: '1 46469U 20066A   25110.45826591  .00000025  00000-0  96784-4 0  9997',
    tle2: '2 46469  65.9963 187.7873 0009304 301.6522  58.3638 13.83374632231318'
  },
  {
    name: 'HAIYANG 1C',
    tle1: '1 43609U 18068A   25110.45163331  .00000304  00000-0  12415-3 0  9993',
    tle2: '2 43609  98.5643 182.7302 0009066 267.3309  92.6838 14.34142709346490'
  }
]
