from PyQt5.QtCore import pyqtSignal, Qt, QEvent
from PyQt5.QtGui import QPalette, QPainter, QBrush, QColor, QPen
from PyQt5.QtWidgets import QWidget, QApplication

from kit.component.kit_button import KitButton


class KitOverlay(QWidget):
    """
    遮罩层
    show() 打开遮罩层，同时置于顶层，如果需要其他组件在最上面，记得调用该组件的raise_()函数
    close() 关闭遮罩层
    setClosePolicy() 设置关闭策略
    """
    CloseOnClicked = 0
    CloseOnEscape = 1
    """
    关闭策略
    CloseOnClicked: 点击遮罩层关闭
    CloseOnEscape: 必须调用close函数才能关闭
    """

    clicked = pyqtSignal()
    resized = pyqtSignal()

    def __init__(self, parent=None):
        super(KitOverlay, self).__init__(parent)
        self._parent = parent
        self.close_policy = KitOverlay.CloseOnEscape

        self.__init_widget()
        self.__init_slot()
        self.__init_qss()

    def __init_widget(self):
        self.setMouseTracking(False)
        self.installEventFilter(self)
        self.hide()

    def __init_slot(self):
        self.clicked.connect(lambda: self.close() if self.close_policy == KitOverlay.CloseOnClicked else None)

    def __init_qss(self):
        palette = QPalette(self.palette())
        palette.setColor(palette.Background, Qt.transparent)
        self.setPalette(palette)

    def __init_parent(self):
        # 用于获取当前应用的最上层的组件，可以理解为主窗口
        widget = QApplication.activeWindow()
        if widget is None:
            widget = self.window()
        if self._parent == widget:
            return
        self._parent = widget
        self.setParent(self._parent)

    def setClosePolicy(self, policy: CloseOnClicked or CloseOnEscape):
        self.close_policy = policy

    def paintEvent(self, event):
        if self._parent is not None:
            self.resize(self._parent.size())
        painter = QPainter()
        painter.begin(self)
        painter.setRenderHint(QPainter.Antialiasing)
        painter.fillRect(event.rect(), QBrush(QColor(0, 0, 0, 127)))
        painter.setPen(QPen(Qt.NoPen))

    def eventFilter(self, obj, e):
        if e.type() == QEvent.MouseButtonPress:
            self.clicked.emit()
        elif e.type() == QEvent.MouseButtonDblClick:
            return True
        elif e.type() == QEvent.MouseMove and e.buttons() == Qt.LeftButton:
            return True
        return super().eventFilter(obj, e)

    def show(self):
        self.focusWidget()
        self.__init_parent()
        self.raise_()
        super().show()

    def resizeEvent(self, a0) -> None:
        super().resizeEvent(a0)
        self.resized.emit()


if __name__ == "__main__":
    from PyQt5.QtWidgets import QApplication, QWidget, QVBoxLayout
    from PyQt5.QtGui import QFontDatabase
    from config import config
    import sys
    QApplication.setAttribute(Qt.AA_EnableHighDpiScaling)
    QApplication.setAttribute(Qt.AA_UseHighDpiPixmaps)

    app = QApplication(sys.argv)
    qss = config.init_qss()
    app.setStyleSheet(qss)
    fontId = QFontDatabase.addApplicationFont("assets/font/iconfont.ttf")
    fontName = QFontDatabase.applicationFontFamilies(fontId)[0]

    main = QWidget()
    layout = QVBoxLayout()
    main.setLayout(layout)
    btn = KitButton('open overlay')
    overlay = KitOverlay()
    overlay.setClosePolicy(KitOverlay.CloseOnClicked)
    layout.addWidget(btn)
    btn.clicked.connect(lambda: [overlay.show(), btn.raise_()])

    main.show()
    sys.exit(app.exec_())