import * as Cesium from 'cesium'
/**
 * @class 基于Entity.Point封装的点
 * @classdesc
 */
class Point {
  /**
   * 点的构造函数
   * @param {Object} map - viewer对象
   * @param {String} [id] - 点对象的ID
   * @param {Array} [position] - 点的位置信息
   * @param {Boolean} [visible=true] - 点的显隐性
   * @param {String} [color='rgb(255,0,0)'] - 点的颜色
   * @param {Number} [alpha=1.0] - 点的不透明度
   * @param {String} [name='point'] - 点的名称
   * @param {Boolean} [nameVisible=true] - 点名称的可见性
   * @param {Number} [nameSize=14] - 点名称的大小
   * @param {Array} [nameOffset=[10,0]] - 点名称的偏移
   * @param {String} [modelUrl] - 模型地址
   * @param {Boolean} [modelVisible=false] - 模型显隐性
   * @param {Number} [modelScale=1] - 模型缩放尺寸
   * @param {Number} [heading=0] - 方位角
   * @param {Number} [pitch=0] - 俯仰角
   * @param {Number} [roll=0] - 滚动角
   * @param {String} [markUrl] - 标记模型地址
   * @param {String} [markScale] - 标记大小
   * @param {String} [markVisible] - 标记显隐性
   */
  constructor(map, options = {}) {
    const viewer = map
    this.entityCollection = viewer.entities; 
    const POINT = {
      id: null,
      position: null,
      visible: true,
      pixelSize:10,
      color: Cesium.Color.fromBytes(255,0,0),
      alpha: 1.0,
      name: 'point',
      nameVisible: true,
      nameSize: 14,
      nameOffset: [10, 10],
      modelUrl: '',
      modelVisible: false,
      modelScale: 1,
      heading: 0,
      pitch: 0,
      roll: 0,
      markUrl: '',
      markScale: 10,
      markVisible:false,
    };
    const finalOptions = { ...POINT, ...options };
    const { position, id, ...pointOptions } = finalOptions;

    this.entity = viewer.entities.add({
      id: id,
      position: position,
      point: pointOptions.visible ? {
        pixelSize: pointOptions.pixelSize,
        color: Cesium.Color.clone(pointOptions.color).withAlpha(pointOptions.alpha)
      }: undefined,
      label: pointOptions.nameVisible ? {
        text: pointOptions.name,
        font: `${pointOptions.nameSize}px sans-serif`,
        fillColor: Cesium.Color.WHITE,
        pixelOffset: new Cesium.Cartesian2(...pointOptions.nameOffset),
      } : undefined,
      model: pointOptions.modelVisible ? {
        uri: pointOptions.modelUrl,
        scale: pointOptions.modelScale,
        minimumPixelSize: 640,
        maximumScale: 20000,
        modelMatrix: Cesium.Transforms.headingPitchRollToFixedFrame(
          position,
          new Cesium.HeadingPitchRoll(
            Cesium.Math.toRadians(pointOptions.heading),
            Cesium.Math.toRadians(pointOptions.pitch),
            Cesium.Math.toRadians(pointOptions.roll)
          )
        ),
      } : undefined,
      billboard: pointOptions.markVisible ? {
        image: pointOptions.markUrl,
        scale: pointOptions.markScale
      } : undefined
    });
  }

  /**
   * 获取查询时刻点所在位置
   * @param {Cesium.JulianDate} time 查询时刻
   * @returns {Cesium.Cartesian3} 点的位置
   */
  getPosition(time) {
    if (!this.entity || !this.entity.position) {
      throw new Error('点实体或位置未定义');
    }
    const position = this.entity.position.getValue(time);
     return position;
  }

  /**
   * 设置点的属性
   *  @param {Object} options 点的属性
   */
  setProperty(options) {
    if (!this.entity) {
      throw new Error('点实体未定义');
    }
  if (options.position) {
    this.entity.position = Cesium.Cartesian3.fromDegrees(options.position.longitude, options.position.latitude,      options.position.height || 0);
  }
  if (options.color) {
    this.entity.point.color = options.color;
  }
  if (options.label) {
    this.entity.label = new Cesium.LabelGraphics({
      text: options.label.text || '',  // 默认为空字符串
      font: options.label.font || '14px sans-serif',  // 默认字体
      fillColor: Cesium.Color.fromCssColorString(options.label.fillColor || '#FFFFFF'),  // 默认白色
      pixelOffset: new Cesium.Cartesian2(
        options.label.pixelOffset ? options.label.pixelOffset.x : 0,
        options.label.pixelOffset ? options.label.pixelOffset.y : 0
      )
    });
  }
  }
  
  /**
   * 从场景中移除点
   */
  removeFromScene() {
  if (!this.entity) {
    throw new Error('点实体未定义');
  }
  this.entityCollection.remove(this.entity);
  this.entity = null;
  }
}
export default Point;