import {uuid} from "vue3-uuid";


export var rules :Record<string, any>[]= []
/**
 * 添加规则
 * @param name  规则名称
 * @param inputs 输入参数 [{key:-1 ,valueKey:'thick',testData:""}]  key表示节点在diagram中的唯一id，valueKey表示节点的属性key
 * @param func  规则函数
 */
export function addRule(name:string, inputs:any[], func:Function){
    rules.push({
        id: uuid.v4(),
        name: name,
        inputs: inputs,
        func: func,
        enable:false,
    })
}

/**
 * 更新规则
 * @param id    规则id
 * @param name  规则名称
 * @param inputs 输入参数
 * @param func  规则函数
 */
export function updateRule(id:string, name:string, inputs:[], func:Function){
    let ruleIndex = rules.findIndex((item:any) => item.id == id)
    if (ruleIndex !== -1){
        rules[ruleIndex].name = name
        rules[ruleIndex].inputs = inputs
        rules[ruleIndex].func = func
    }
}

/**
 * 删除规则
 * @param id
 */
export function deleteRuleById(id:string){
    rules.splice(rules.findIndex((item:any) => item.id == id), 1)
}


/**
 * 运行规则
 * @param rule 规则
 * @param deviceMap 设备数据
 */
export function runRule(rule:any, deviceMap:Record<string, any>){
    let inputData = []
    let runFunc = null
    if (rule){
        for (let input of rule.inputs){
            inputData.push(deviceMap[input.key][input.valueKey])
        }
        runFunc = new Function( "return " + rule.func)
        try {
            runFunc()(...inputData)
        }catch (e){
            alert(e)
        }
    }
}


/**
 * 函数“runAllEnableRule”在给定的设备映射上运行所有启用的规则。
 * @param deviceMap - 将设备名称映射到其相应信息或数据的记录对象。
 */
export function runAllEnableRule(deviceMap:Record<string, any>){
    rules.forEach((item:any) => {
        if(item['enable'].toString() === 'true') {
            runRule(item, deviceMap)
        }
    })
}

/**
 * 函数“runRuleById”通过 ID 查找规则并使用给定的设备映射运行它。
 * @param {string} id - “id”参数是一个字符串，表示规则的唯一标识符。它用于在“rules”数组中查找规则。
 * @param deviceMap - “deviceMap”参数是一个记录对象，它将设备名称映射到其相应的信息或状态。它用于提供运行规则所需的数据。
 */
export function runRuleById(id:string, deviceMap:Record<string, any>){
    let rule = rules.find((item:any) => item.id == id)
    runRule(rule, deviceMap)
}

/**
 * 函数“testRule”将规则对象作为输入，从规则中提取测试数据，从规则的函数字符串创建一个新函数，然后使用测试数据作为参数执行该函数。
 * @param {any} rule - “rule”参数是包含有关特定规则的信息的对象。
 */
export function testRule(rule:any){
    let inputData = []
    let runFunc = null
    if (rule){
        for (let input of rule.inputs){
            inputData.push(input['testData'])
        }
        runFunc = new Function( "return " + rule.func)
        try {
            runFunc()(...inputData)
        }catch (e){
            alert(e)
        }
    }
}

/**
 * 函数“testRuleById”通过 ID 查找规则，然后测试该规则。
 * @param {string} id - “id”参数是一个字符串，表示规则的唯一标识符。
 */
export function testRuleById(id:string){
    let rule = rules.find((item:any) => item.id == id)
    testRule(rule)
}