import {uuid} from "vue3-uuid";
import * as fengari from 'fengari-web';
import {ElNotification} from "element-plus";

export class RuleScriptType {
    static Lua = 'lua'
    static JS = 'js'
}

export const RuleActionType = [
    {key:'success', name:"正常" },
    {key:'info', name:"通知" },
    {key:'warning', name:"警告" },
    {key:'error', name:"故障" },
]

/**
 * {
 *     id: string // 规则id 也是函数名
 *     type: string // 规则语法类型 例如：lua,js等 暂定为只有lua
 *     inputs:[
 *       key: 对应设备的唯一id,
 *       valueKey: "name", 指定设备下的属性key,
 *       testData: 测试数据
 *     ] 输入的参数列表
 *     func: string // 规则函数
 *     enable: boolean // 是否启用
 *     name: string // 规则名称
 *     description: string // 规则描述
 *     actions: [
 *         type: 通知的类型 例如：success，info,warning,error
 *         content: string, 通知的内容
 *     ] // 规则动作 根据用户填写的规则的返回值来确定触发的效果
 * }
 */
export var rules: Record<string, any>[] = []

/**
 * 添加规则
 * @param name  规则名称
 * @param description 规则描述
 * @param inputs 输入参数
 * @param func  规则函数
 * @param actions 规则动作
 */
export function addRule(name: string, description: string, inputs: any[], func: Function, actions:any[]=[]) {
    const ruleId =uuid.v4()
    rules.push({
        id: ruleId,
        name: name,
        inputs: inputs,
        description: description,
        func: func,
        enable: false,
        type: RuleScriptType.Lua, // 暂定为只有lua
        actions: actions
    })
}

/**
 * 更新规则
 * @param id    规则id
 * @param name  规则名称
 * @param description 规则描述
 * @param inputs 输入参数
 * @param func  规则函数
 */
export function updateRule(id: string, name: string, description: string, inputs: [], func: Function) {
    let ruleIndex = rules.findIndex((item: any) => item.id == id)
    if (ruleIndex !== -1) {
        rules[ruleIndex].name = name
        rules[ruleIndex].inputs = inputs
        rules[ruleIndex].description = description
        rules[ruleIndex].func = func
    }
}

/**
 * 删除规则
 * @param id
 */
export function deleteRuleById(id: string) {
    rules.splice(rules.findIndex((item: any) => item.id == id), 1)
}


/**
 * 运行规则
 * @param rule 规则
 * @param deviceMap 设备数据
 */
export function runRule(rule: any, deviceMap: Record<string, any>) {
    let inputData = []
    let result = -1
    if (rule) {
        for (let input of rule.inputs) {
            inputData.push(deviceMap[input.key][input.valueKey])
        }
        switch (rule['type']) {
            case RuleScriptType.Lua:
                result = runLua(rule, inputData)
                break
            case RuleScriptType.JS:
                result = runJS(rule, inputData)
                break
            default:
                alert('no such rule type' + rule['type'])
        }
    }
    runRuleAction(rule, result)

    return result
}


/**
 * 函数“runAllEnableRule”在给定的设备映射上运行所有启用的规则。
 * @param deviceMap - 将设备名称映射到其相应信息或数据的记录对象。
 */
export function runAllEnableRule(deviceMap: Record<string, any>) {
    rules.forEach((item: any) => {
        if (item['enable'].toString() === 'true') {
            runRule(item, deviceMap)
        }
    })
}

/**
 * 函数“runRuleById”通过 ID 查找规则并使用给定的设备映射运行它。
 * @param {string} id - “id”参数是一个字符串，表示规则的唯一标识符。它用于在“rules”数组中查找规则。
 * @param deviceMap - “deviceMap”参数是一个记录对象，它将设备名称映射到其相应的信息或状态。它用于提供运行规则所需的数据。
 */
export function runRuleById(id: string, deviceMap: Record<string, any>) {
    let rule = rules.find((item: any) => item.id == id)
    runRule(rule, deviceMap)
}

/**
 * 函数“testRule”将规则对象作为输入，从规则中提取测试数据，从规则的函数字符串创建一个新函数，然后使用测试数据作为参数执行该函数。
 * @param {any} rule - “rule”参数是包含有关特定规则的信息的对象。
 */
export function testRule(rule: any) {
    let inputData = []
    let result = -1
    if (rule) {
        for (let input of rule.inputs) {
            inputData.push(input['testData'])
        }
        switch (rule['type']) {
            case RuleScriptType.Lua:
                result = runLua(rule, inputData)
                break
            case RuleScriptType.JS:
                result = runJS(rule, inputData)
                break
            default:
                alert('no such rule type' + rule['type'])
        }
    }
    runRuleAction(rule, result)
    return result
}

/**
 * 函数“testRuleById”通过 ID 查找规则，然后测试该规则。
 * @param {string} id - “id”参数是一个字符串，表示规则的唯一标识符。
 */
export function testRuleById(id: string) {
    let rule = rules.find((item: any) => item.id == id)
    testRule(rule)
}

export function runRuleAction(rule:any, result:any){
    try{
        result = Number(result)
    }catch(e){
        alert("脚本的返回值必须是数字")
    }
    if (result>=0 && result<rule.actions.length){
        let ruleType = rule.actions[result].type
        let ruleContent = rule.actions[result].content
        ElNotification({
            title: rule['name'],
            message: ruleContent,
            type: ruleType,
        })
    }
}

export function runJS(rule: any, inputData: any) {
    const funcBody = "(data)=>{ "+ rule['func']+"}"
    const runFunc = new Function("return " + funcBody)
    try {
        return runFunc()(inputData)
    } catch (e) {
        alert(e)
    }
}

export function runLua(rule: any, inputData: any) {
    try {
        // 创建Lua状态
        let codeString = `
        return function (data)
            ${rule['func']}
        end
        `

        let luaFunction = fengari.load(codeString)
        return luaFunction().call(inputData)

    } catch (e) {
        alert(e)
    }
}


