import * as math from 'mathjs'

/* DeviceType 类定义不同类型设备的常量。 */
export class DeviceType{
    static CONTAINER = 'container';
    static PUMP = 'pump';
}

export const thingFactoryData:Record<string, number> = {
    '0.02': 1,
    '0.04': 1,
    '0.06': 1,
    '0.08': 1,
    '0.1': 1,
    '0.12': 2,
    '0.14': 10,
    '0.16': 50,
    '0.18': 130,
    '0.2': 210,
    '0.25': 290,
    '0.3': 400,
    '0.35': 550,
    '0.4': 700,
    '0.45': 810,
    '0.5': 970,
    '0.55': 1130,
    '0.6': 1250,
    '0.65': 1400,
}
//计算设备失效可能性
//失效概率 = 同类设备失效平均概率 * 管理系统评价系数 * 设备修正系数 * 超标缺陷影响系数

export function getDeviceFailurePossibility(averageFailurePossibility:math.BigNumber, managementSystemEvaluationCoefficient:math.BigNumber, deviceCorrectionCoefficient:math.BigNumber, exceedingDefectInfluenceCoefficient:math.BigNumber){
    return math.chain(averageFailurePossibility).multiply(managementSystemEvaluationCoefficient).multiply(deviceCorrectionCoefficient).multiply(exceedingDefectInfluenceCoefficient).done()
}

// 计算同类设备平均失效概率
export function getAverageFailurePossibility(deviceType:string){
    if (deviceType == DeviceType.CONTAINER){
        return math.bignumber(8e-6)
    }else if(deviceType == DeviceType.PUMP){
        return math.bignumber(8e-6)
    }else{
        return math.bignumber(0)
    }

}

// 计算管理系统评价系数
export function getManagementSystemEvaluationCoefficient(){
    return math.bignumber(0.1)
}

// 计算设备修正系数
export function getDeviceCorrectionCoefficient(deviceType: string = DeviceType.CONTAINER, trd: number | string, a: number | string, r: number | string, tmin: number | string, CA: number | string){
    // 计算单个设备的减薄因子
    let thinningFactor
    if (deviceType === DeviceType.CONTAINER){
        thinningFactor = getContainerThinningFactor(trd, a, r, tmin, CA)
    }else if(deviceType === DeviceType.PUMP){
        thinningFactor = getPumpThinningFactor(trd, a, r, tmin, CA)
    }else{
        thinningFactor = math.bignumber(0)
    }
    let thinningData = convertThinningFactor(thinningFactor)
    return math.chain(25).add(thinningData).done() as math.BigNumber
}

// 计算超标缺陷影响系数
export function getExceedingDefectInfluenceCoefficient(){
    return math.bignumber(1.0)
}

/**
* @param trd    检测厚度 mm
* @param a      使用年限 year
* @param r      腐蚀速率 mm/year
* @param tmin   计算厚度 mm
* @param CA     腐蚀裕量 mm
* @return       容器减薄次因子
*/

export function getContainerThinningFactor(trd: number|string, a: number|string, r:number|string, tmin:number|string, CA:number|string) {
    let m_trd = math.bignumber(trd)
    let m_a = math.bignumber(a)
    let m_r = math.bignumber(r)
    let m_tmin = math.bignumber(tmin)
    let m_CA = math.bignumber(CA)

    let molecular = math.chain(m_trd).subtract(math.chain(m_a).multiply(m_r).done()).done()
    let denominator = math.chain(m_tmin).add(m_CA).done()
    let division = math.chain(molecular).divide(denominator).done()
    let result = math.chain(math.bignumber(1)).subtract(division).done() as math.BigNumber
    if (math.compare(result, 0) == -1) {
        result = math.bignumber(0);
    }
    return result
}
/**
 * @param trd    检测厚度 mm
 * @param a      使用年限 year
 * @param r      腐蚀速率 mm/year
 * @param t      名义厚度 mm
 * @param CA     腐蚀裕量 mm
 * @return       管道减薄次因子
 */
export function getPumpThinningFactor(trd:number|string, a:number|string, r:number|string, t:number|string, CA:number|string){
    let m_trd = math.bignumber(trd),
        m_a = math.bignumber(a),
        m_r = math.bignumber(r),
        m_t = math.bignumber(t),
        m_CA = math.bignumber(CA)

    let molecular = math.chain(m_t).subtract(m_trd).add(math.chain(m_a).multiply(m_r).done()).done()
    let denominator = math.chain(m_t).subtract(m_CA).done()
    let result = math.chain(molecular).divide(denominator).done() as math.BigNumber
    return result
}

export function convertThinningFactor(thinningFactor:math.BigNumber){
    let m_thinningFactor = thinningFactor
    let keys = Object.keys(thingFactoryData)

    if (m_thinningFactor < math.bignumber(keys[0])){
        return math.bignumber(thingFactoryData[keys[0]])
    }
    if (m_thinningFactor > math.bignumber(keys[keys.length-1])){
        return math.bignumber(thingFactoryData[keys[keys.length-1]])
    }

    //找到m_thinningFactor 夹在中间的地方
    for(let i=0;i<keys.length-1; i++){
        if (math.compare(m_thinningFactor, Number(keys[i])) == 1 && math.compare(m_thinningFactor, Number(keys[i+1])) == -1){
            let a = math.bignumber(thingFactoryData[keys[i]])
            let b = math.bignumber(thingFactoryData[keys[i+1]])
            let c = math.subtract(m_thinningFactor, a)
            let d = math.subtract(b, m_thinningFactor)
            if (math.compare(c, d) == 1){
                return b
            }else{
                return a
            }
        }
    }
    return math.bignumber(1)
}