import go from 'gojs';

export var GO = go.GraphObject.make;

/**
 * 获取节点的所有连接,存储为一个Map。{key:{next:[], data:{}}}
 */
export function getMapData(myDiagram:any) {

    let model = myDiagram.model
    let nodeDataArray = model.nodeDataArray
    let linkDataArray = model.linkDataArray
    let nodeMap: Record<any, any> = {}
    nodeDataArray.forEach((item:any)=>{
        nodeMap[item.key] = {next:[], data:item}
    })
    linkDataArray.forEach((item:any)=>{
        let node = null;
        if (item.from && item.to){
            node = nodeMap[item.from]
            node.next.push(item.to)
        }
    })
    return nodeMap
}

/**
 * 获取所有选择的节点
 * @param {go.Diagram} myDiagram
 * @return [{},{}]
 */

export function getSelectedNodes(myDiagram:go.Diagram) {
    let selectObjIterator = myDiagram.selection.iterator;
    let resultNodes = [];
    while (selectObjIterator.next()) {
        let selectedObj = selectObjIterator.value;
        if (selectedObj instanceof go.Node) {
            resultNodes.push(selectedObj.data);
        }
    }
    return resultNodes;
}

/**
 * 获取选中的两点间的所有路径
 * @param startKey
 * @param endKey
 * @param myDiagram
 */
export function findNodesAllWayBetween(startKey:number, endKey:number, myDiagram:go.Diagram) {
    // 保存所有路径
    let paths = [[startKey]],
        // 搜索是否完成
        flag:boolean = false,
        nodeLinkMap = getMapData(myDiagram);

    while(!flag){
        // 保存当前路径的节点数量
        let beforePathsNode = 0;
        // 保存扩展后路径的节点数量
        let afterPathsNode = 0;
        // 计算扩展前的总路径节点数量
        for (let path of paths){
            beforePathsNode += path.length
        }
        for (let path of paths){
            // 获取当前路径的最后一个节点
            let lastNode = path[path.length-1]
            // 如果最后一个节点是终点，则跳过，无需扩展
            if (lastNode == endKey){
                continue
            }
            // 获取当前路径最后一个节点的下一个节点
            let nextNode = nodeLinkMap[lastNode].next
            // 如果没有下一个节点，则跳过，无需扩展
            if (nextNode.length ==0){
                continue
            }
            // 如果只有一个节点，则将下一个节点直接加入当前路径
            if (nextNode.length ==1 ){
                path.push(nextNode[0])
            }
            // 如果有多个节点，则将当前路径分裂为多个路径
            else{
                for (let nextNodeKey of nextNode){
                    // 如果下一个节点已经在路径中，则跳过，无需扩展，避免循环
                    if (nextNodeKey in path){
                        continue
                    }
                    let newPath = path.slice(0)
                    newPath.push(nextNodeKey)
                    paths.push(newPath)
                }
                // 删除没有扩展的原路径
                paths = paths.filter((item:any) => item != path
                )
            }
        }
        // 计算扩展后的总路径节点数量
        for (let path of paths){
            afterPathsNode += path.length
        }
        // 如果paths内元素数量没有增加，则搜索完成
        if (beforePathsNode == afterPathsNode){
            flag = true
            // 删除没有找到终点的路径
            paths = paths.filter((item:any) => item[item.length-1] == endKey)
        }
    }
    return paths
}

/**
 * 根据 from 和 to 高亮某条线
 */
export function highlightLink(from:number, to:number, myDiagram:any){
    let model = myDiagram.model
    let linkDataArray = model.linkDataArray
    linkDataArray.forEach((item:any)=>{
        if (item.from == from && item.to == to){
            model.set(item, "color", "red")
        }
    })
}

export function cancelHighlightLink(myDiagram:any){
    let model = myDiagram.model
    let linkDataArray = model.linkDataArray
    linkDataArray.forEach((item:any)=>{
        model.set(item, "color", "black")
    })
}

/**
 * 这里只以计算压力数据为例。
 */
export function computeData(pathNodesMap:any, analyzeResult:any){
    let nodesDataMap:Record<any, any> = {}
    let offset = 0.1
    console.log("compute", pathNodesMap);
    for (let key in pathNodesMap){
        let path = pathNodesMap[key]
        for (let node of path.nodes){
            if (node.data.key in nodesDataMap){
                let currentNodeIndex = path.nodes.indexOf(node)
                if (currentNodeIndex == 0){
                    continue
                }
                let lastNode = path.nodes[currentNodeIndex-1]
                nodesDataMap[node.data.key]['predict'] += lastNode.data.press * path.weight
            }
            else{
                nodesDataMap[node.data.key] = {'predict': 0, 'real': node.data.press}
                let currentNodeIndex = path.nodes.indexOf(node)
                if (currentNodeIndex == 0){
                    nodesDataMap[node.data.key]['predict'] = node.data.press
                    continue
                }
                let lastNode = path.nodes[currentNodeIndex-1]
                nodesDataMap[node.data.key]['predict'] += lastNode.data.press * path.weight
            }
        }
    }
    analyzeResult.value = {}
    for (let key in nodesDataMap){
        let value = nodesDataMap[key]
        if(value['predict'] < value['real']*(1-offset)){
            analyzeResult.value[key] = "压力过高"
        }
        else if(value['predict'] > value['real']*(1+offset)){
            analyzeResult.value[key] = "压力过低"
        }
        else{
            analyzeResult.value[key] = "压力正常"
        }
    }
    console.log(analyzeResult.value)
    return nodesDataMap
}
