from django.db import models
from django.apps import apps
from django.contrib.auth.models import UserManager, AbstractUser
from django.contrib.auth.hashers import make_password, check_password
from django.core import signing, signals

# Create your models here.


class MyUserManager(UserManager):

    def _create_user(self, username, email, password, **extra_fields):
        """
        Create and save a user with the given username, email, and password.
        """
        if not username:
            raise ValueError('The given username must be set')
        email = self.normalize_email(email)
        # Lookup the real model class from the global app registry so this
        # manager method can be used in migrations. This is fine because
        # managers are by definition working on the real model.
        GlobalUserModel = apps.get_model(self.model._meta.app_label, self.model._meta.object_name)
        username = GlobalUserModel.normalize_username(username)
        algorithm = extra_fields.pop('algorithm', 'default')
        user = self.model(username=username, email=email, **extra_fields)
        user.password = make_password(password, hasher=algorithm)
        user.save(using=self._db)
        return user


class User(AbstractUser):
    objects = MyUserManager()
    
    def check_password(self, raw_password):
        """
        Return a boolean of whether the raw_password was correct. Handles
        hashing formats behind the scenes.
        """
        # def setter(raw_password):
        #     self.set_password(raw_password)
        #     # Password hash upgrades shouldn't be considered password changes.
        #     self._password = None
        #     self.save(update_fields=["password"])
        return check_password(raw_password, self.password)

    class Meta(AbstractUser.Meta):
        swappable = 'AUTH_USER_MODEL'


class Note(models.Model):
    id = models.AutoField(primary_key=True)
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    note = models.TextField()
    date_time = models.DateTimeField(auto_now_add=True)
    update_time = models.DateTimeField(auto_now=True)
