import json

from device_data_op.models import TableAllDevCmdDefine, TableDevCmdNamePoll
from device_data_op.serializers import TableAllDevCmdDefineSerializer, TableDevCmdNamePollSerializer
from django.db.models import QuerySet
from rest_framework.response import Response
from rest_framework import status
from .models import (AllDevCmdDefineAndVersion, AllProtocolDefinAndVersion,
                     AllProtocolVersion, CurrentDevVersion)
from .serializers import (AllDevCmdDefineAndVersionSerializer, AllProtocolDefinAndVersionSerializer,
                         AllProtocolVersionSerializer, CurrentDevVersionSerializer)


INIT_VERSION = "init"


def init_protocol_version_manage(protocol_name: str) -> None:
    """
    将协议版本信息初始化，即先将 device 已经在用的表的数据添加到数据库表中
    
    :param protocol_name: 协议名
    """
    protocol_cmds = TableDevCmdNamePoll.objects.filter(protocol_name=protocol_name).all()
    protocol_cmds_serializer = TableDevCmdNamePollSerializer(protocol_cmds, many=True)
    all_protocol_objects: list[AllProtocolDefinAndVersion] = []
    for protocol in protocol_cmds_serializer.data:
        protocol['version'] = json.dumps([INIT_VERSION])
        del protocol['id']
        all_protocol_objects.append(AllProtocolDefinAndVersion(**protocol))
    AllProtocolDefinAndVersion.objects.bulk_create(all_protocol_objects)

    cmd_fields: list[QuerySet] = []
    for protocol in protocol_cmds:
        cmd_fields.extend(TableAllDevCmdDefine.objects.filter(cmd_name=protocol.cmd_name).all())
    cmd_fields_serializer = TableAllDevCmdDefineSerializer(cmd_fields, many=True)
    all_cmd_fields_objects: list[AllDevCmdDefineAndVersion] = []
    for cmd_field in cmd_fields_serializer.data:
        cmd_field['version'] = json.dumps([INIT_VERSION])
        del cmd_field['id']
        all_cmd_fields_objects.append(AllDevCmdDefineAndVersion(**cmd_field))
    AllDevCmdDefineAndVersion.objects.bulk_create(all_cmd_fields_objects)
    
    version_path = {
        "version": INIT_VERSION
    }
    AllProtocolVersion.objects.create(protocol_name=protocol_name, 
                                      version_paths=json.dumps([version_path]))
    
    CurrentDevVersion.objects.create(protocol_name=protocol_name,
                                     version=INIT_VERSION)


def update_device_protocol_and_cmds(protocol_name: str, version: str) -> Response:
    """
    根据协议名和版本号更新协议版本信息
    
    通过协议名和版本号将 AllProtocolDefinAndVersion 和 AllDevCmdDefineAndVersion 中符合
    要求的数据更新到 TableDevCmdNamePoll 和 TableAllDevCmdDefine 中

    :param protocol_name: 协议名
    :param version: 版本号

    :return: 当前版本，协议相关的数据
    """
    # 更新版本号
    CurrentDevVersion.objects.filter(protocol_name=protocol_name).update(version=version)

    # 更新协议版本信息
    all_protocol = AllProtocolDefinAndVersion.objects.filter(protocol_name=protocol_name).all()
    current_version_protocols = [protocol for protocol in all_protocol 
                                 if version in json.loads(protocol.version)]
    cmd_fields = []
    for current_version_protocol in current_version_protocols:
        cmd_name = current_version_protocol.cmd_name
        cmd_fields.extend([item for item in AllDevCmdDefineAndVersion.objects.filter(cmd_name=cmd_name).all()
                           if version in json.loads(item.version)])
    
    current_version_protocol_serializer = AllProtocolDefinAndVersionSerializer(current_version_protocols, many=True)
    cmd_fields_serializer = AllDevCmdDefineAndVersionSerializer(cmd_fields, many=True)

    # 更新 device 所使用的命令表数据
    try:
        TableAllDevCmdDefine.objects.filter(cmd_name__in=[cmd.cmd_name 
                                                          for cmd in TableDevCmdNamePoll.objects.filter(protocol_name=protocol_name).all()]).delete()
        TableAllDevCmdDefine.objects.bulk_create([TableAllDevCmdDefine(
                                                **{k: v for k, v in cmd_field.items() if k not in ['id', 'version']}) 
                                                for cmd_field in cmd_fields_serializer.data])
    except Exception as e:
        print(e)
        return Response(status=status.HTTP_500_INTERNAL_SERVER_ERROR)
    # 更新 device 所使用的协议表数据
    try:
        TableDevCmdNamePoll.objects.filter(protocol_name=protocol_name).delete()
        TableDevCmdNamePoll.objects.bulk_create([TableDevCmdNamePoll(
                                                **{k: v for k, v in protocol.items() if k not in ['id', 'version']}) 
                                                for protocol in current_version_protocol_serializer.data])
    except Exception as e:
        print(e)
        return Response(status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    # 返回数据
    cmds = []
    for protocol in current_version_protocol_serializer.data:
        protocol['fields'] = sorted([item for item in cmd_fields_serializer.data 
                                     if item['cmd_name'] == protocol['cmd_name']], 
                                     key=lambda item: item['fieldindex'])
        cmds.append(protocol)

    res_data = {
        'current_version': version,
        'protocol': protocol_name,
        'cmds': cmds
    }

    return Response(data=res_data, status=status.HTTP_200_OK)
        





