from rest_framework.decorators import api_view
from rest_framework.response import Response
from rest_framework import status, mixins
from rest_framework.viewsets import GenericViewSet

from .models import (SimulateDeviceCommunicationParameter, DeviceCommunicationParameter)
from protocol_version_manage.models import (AllProtocolVersion, AllProtocolDefinAndVersion,
                                            AllDevCmdDefineAndVersion)
from device_data_op.models import TableXproAllDevinfo
from .serializers import (SimulateDeviceCommunicationParameterSerializer,
                          DeviceCommunicationParameterSerializer)


@api_view(['GET'])
def get_protocol_names(request):
    """
    获取所有协议，并返回
    """
    protocol_names = AllProtocolVersion.objects.all()
    res_data = ([{'value': protocol_name.protocol_name, 'label': protocol_name.protocol_name} 
                 for protocol_name in protocol_names])

    return Response(data=res_data, status=status.HTTP_200_OK)


@api_view(['POST'])
def get_protocol_field_names(request):
    """
    获取指定协议下的指令集
    """
    protocol_name = request.data.get('protocol_name')
    if protocol_name is None or protocol_name == '': 
        return Response(status=status.HTTP_400_BAD_REQUEST)

    # 获取该协议下的指令集合
    cmds = AllProtocolDefinAndVersion.objects.filter(protocol_name=protocol_name, cmd_type='RX').all()
    cmd_set = set([cmd.cmd_name for cmd in cmds])

    # 获取指令集合下的字段
    fields = AllDevCmdDefineAndVersion.objects.filter(cmd_name__in=cmd_set, cmd_type='RX').all()
    field_name_set = set([field.fieldname for field in fields])

    res_data = ([field_name for field_name in field_name_set])
    return Response(data=res_data, status=status.HTTP_200_OK)


@api_view(['POST'])
def set_communication_to_devinfo_table(request):
    """
    将设备通信参数或模拟设备通信参数设置到设备信息表
    """
    type = request.data.get('type')
    if type is None or type == '':
        return Response(status=status.HTTP_400_BAD_REQUEST)
    
    if type == 'simulate':
        communications = SimulateDeviceCommunicationParameter.objects.all()
    else:
        communications = DeviceCommunicationParameter.objects.all()
    
    # 清空设备信息表
    # TableXproAllDevinfo.objects.all().delete()

    TableXproAllDevinfo.objects.bulk_create(
        [TableXproAllDevinfo(sta_id=communication.station_id,
                             dev_id=communication.device_id,
                             dev_name=communication.device_name,
                             dev_name_chn=communication.device_name_chn,
                             protocol_name=communication.protocol_name, 
                             cmd_excel_path="null",
                             comunitate_mode=communication.communicate_mode,
                             tcp_ip=communication.tcp_ip,
                             tcp_port=communication.tcp_port,
                             udp_ip_src=communication.udp_ip_src,
                             udp_port_src=communication.udp_port_src,
                             udp_ip_dst=communication.udp_ip_dst,
                             udp_port_dst=communication.udp_port_dst,
                             udpmc_ip="",
                             udpmc_ip_tx="",
                             udpmc_port_tx=0,
                             udpmc_ip_rx="",
                             udpmc_port_rx=0,
                             remarks="")
         for communication in communications])

    return Response(status=status.HTTP_200_OK)


class DeviceCommunicationParameterViewSet(mixins.CreateModelMixin,
                                          GenericViewSet):
    queryset = DeviceCommunicationParameter.objects.all()
    serializer_class = DeviceCommunicationParameterSerializer

