import json

from drf_yasg import openapi
from drf_yasg.utils import swagger_auto_schema
from django.shortcuts import render
from rest_framework.response import Response
from rest_framework.viewsets import ModelViewSet
from rest_framework.decorators import api_view
from rest_framework import status
from protocol_version_manage.models import (CurrentDevVersion, AllDevCmdDefineAndVersion,
                                            AllProtocolDefinAndVersion, AllProtocolVersion)
from .models import (TableAllDevCmdDefine, TableDevCmdNamePoll, 
                     TableSoftLimitAngle, TableXproAllDevinfo)
from .serializers import (TableAllDevCmdDefineSerializer, TableDevCmdNamePollSerializer, 
                          TableSoftLimitAngleSerializer, TableXproAllDevinfoSerializer)
from .utils import tree_data


class TableAllDevCmdDefineView_1(ModelViewSet):
    queryset = TableAllDevCmdDefine.objects.all()
    serializer_class = TableAllDevCmdDefineSerializer

    def list(self, request):
        serializer = self.get_serializer(self.get_queryset(), many=True)
        data = tree_data(serializer.data, 'cmd_name')
        return Response(data)

    def perform_create(self, serializer):
        """
        新增指令字段，给 TableAllDevCmdDefine 表创建记录时，同时给 AllDevCmdDefineAndVersion 表创建
        """
        # 给表创建值
        super().perform_create(serializer)

        current_obj = CurrentDevVersion.objects.filter(protocol_name=serializer.validated_data['protocol_name']).first()
        assert current_obj is not None, "当前协议不存在"

        current_version = current_obj.version
        serializer.validated_data['version'] = current_version
        serializer.validated_data.pop('protocol_name')
        AllDevCmdDefineAndVersion.objects.create(**serializer.validated_data)
    
    def perform_destroy(self, instance):
        """
        删除指令字段，给 TableAllDevCmdDefine 表删除记录时，同时给 AllDevCmdDefineAndVersion 表删除
        这里由于不能直接获取到 protocol_name，所以需要先通过 TableDevCmdNamePoll 获取到 protocol_name，然后再删除
        """
        cmd_name = instance.cmd_name

        cmd_obj = TableDevCmdNamePoll.objects.filter(cmd_name=cmd_name).first()
        assert cmd_obj is not None, "当前指令不存在"

        current_protocol_name = cmd_obj.protocol_name
        current_obj = CurrentDevVersion.objects.filter(protocol_name=current_protocol_name).first()
        assert current_obj is not None, "当前协议不存在"

        current_version = current_obj.version
        AllDevCmdDefineAndVersion.objects.filter(cmd_name=instance.cmd_name, 
                                                 fieldname=instance.fieldname, 
                                                 version=current_version).delete()
        # 更新 fieldindex
        super().perform_destroy(instance)
        fields = self.get_queryset().filter(cmd_name=cmd_name).all()
        for i in range(len(fields)):
            fields[i].fieldindex = i + 1
            fields[i].save()

        fields_ = AllDevCmdDefineAndVersion.objects.filter(cmd_name=cmd_name, version=current_version).all()
        for i in range(len(fields_)):
            fields_[i].fieldindex = i + 1
            fields_[i].save()

    
    def perform_update(self, serializer):
        """
        更新指令字段，给 TableAllDevCmdDefine 表更新记录时，同时给 AllDevCmdDefineAndVersion 表更新
        """
        super().perform_update(serializer)

        current_obj = CurrentDevVersion.objects.filter(protocol_name=serializer.validated_data['protocol_name']).first()
        assert current_obj is not None, "当前协议不存在"

        current_version = current_obj.version
        serializer.validated_data['version'] = current_version
        serializer.validated_data.pop('protocol_name')

        AllDevCmdDefineAndVersion.objects.filter(cmd_name=serializer.validated_data['cmd_name'],
                                                 fieldname=serializer.validated_data['fieldname'],
                                                 version=current_version).update(**serializer.validated_data)


class TableDevCmdNamePollView_1(ModelViewSet):
    queryset = TableDevCmdNamePoll.objects.all()
    serializer_class = TableDevCmdNamePollSerializer

    def list(self, request):
        serializer = self.get_serializer(self.get_queryset(), many=True)
        data = tree_data(serializer.data, 'protocol_name')
        return Response(data)

    def perform_create(self, serializer):
        """
        新增指令，给 TableDevCmdNamePollView 表创建记录时，同时给 AllProtocolDefineAndVersion 表创建
        如果当前协议不存在，则在 CurentDevVersion 表创建记录，同时在 AllProtocolVersion 表创建记录
        """
        super().perform_create(serializer)

        current_obj = CurrentDevVersion.objects.filter(protocol_name=serializer.validated_data['protocol_name']).first()
        if current_obj is None:
            current_obj = CurrentDevVersion.objects.create(protocol_name=serializer.validated_data['protocol_name'], 
                                                           version="init")
            current_version = "init"
            AllProtocolVersion.objects.create(protocol_name=serializer.validated_data['protocol_name'],
                                              version_paths=json.dumps([{"version": "init"}]))
        else:
            current_version = current_obj.version
        serializer.validated_data['version'] = current_version
        AllProtocolDefinAndVersion.objects.create(**serializer.validated_data)
    
    def perform_destroy(self, instance):
        """
        删除指令，给 TableDevCmdNamePollView 表删除记录时，同时给 AllProtocolDefineAndVersion 表删除
        """
        protocol_name = instance.protocol_name
        cmd_name = instance.cmd_name

        current_obj = CurrentDevVersion.objects.filter(protocol_name=protocol_name).first()
        assert current_obj is not None, "当前协议不存在"
        AllProtocolDefinAndVersion.objects.filter(protocol_name=protocol_name,
                                                  cmd_name=cmd_name,
                                                  version=current_obj.version).delete()

        super().perform_destroy(instance)
    
    def perform_update(self, serializer):
        """
        更新指令，给 TableDevCmdNamePollView 表更新记录时，同时给 AllProtocolDefineAndVersion 表更新
        """
        current_obj = CurrentDevVersion.objects.filter(protocol_name=serializer.validated_data['protocol_name']).first()
        assert current_obj is not None, "当前协议不存在"
        serializer.validated_data['version'] = current_obj.version
        AllProtocolDefinAndVersion.objects.filter(protocol_name=serializer.validated_data['protocol_name'],
                                                  cmd_name=serializer.validated_data['cmd_name'],
                                                  version=current_obj.version).update(**serializer.validated_data)
        super().perform_update(serializer)


class TableAllDevCmdDefineView(ModelViewSet):
    queryset = TableAllDevCmdDefine.objects.all()
    serializer_class = TableAllDevCmdDefineSerializer

    def list(self, request):
        serializer = self.get_serializer(self.get_queryset(), many=True)
        data = tree_data(serializer.data, 'cmd_name')
        return Response(data)
    
    def perform_destroy(self, instance):
        """
        删除某个字段，需要将字段的 index 更新
        """
        
        # 获取改字段的 cmd_name
        cmd_name = instance.cmd_name
        super().perform_destroy(instance)
        fields = self.get_queryset().filter(cmd_name=cmd_name).all()

        # 更新字段的 index
        for i in range(len(fields)):
            print(fields[i])
            fields[i].fieldindex = i + 1
            fields[i].save()


class TableDevCmdNamePollView(ModelViewSet):
    queryset = TableDevCmdNamePoll.objects.all()
    serializer_class = TableDevCmdNamePollSerializer

    def list(self, request):
        serializer = self.get_serializer(self.get_queryset(), many=True)
        data = tree_data(serializer.data, 'protocol_name')
        return Response(data)
    

class TableSoftLimitAngleView(ModelViewSet):
    queryset = TableSoftLimitAngle.objects.all()
    serializer_class = TableSoftLimitAngleSerializer


class TableXproAllDevinfoView(ModelViewSet):
    queryset = TableXproAllDevinfo.objects.all()
    serializer_class = TableXproAllDevinfoSerializer


@swagger_auto_schema(method='post', request_body=openapi.Schema(
    type=openapi.TYPE_OBJECT,
    properties={
        'cmds': openapi.Schema(type=openapi.TYPE_OBJECT, properties={
            'protocol_name': openapi.Schema(type=openapi.TYPE_STRING),
            'cmd_name': openapi.Schema(type=openapi.TYPE_STRING),
            'cmd_type': openapi.Schema(type=openapi.TYPE_STRING),
            'encode': openapi.Schema(type=openapi.TYPE_STRING),
            'timing_cmd_cycle_period': openapi.Schema(type=openapi.TYPE_INTEGER),
            'cmd_explain': openapi.Schema(type=openapi.TYPE_STRING),
            'fields': openapi.Schema(type=openapi.TYPE_ARRAY, items=openapi.Items(type=openapi.TYPE_OBJECT, properties={
                "cmd_name": openapi.Schema(type=openapi.TYPE_STRING),
                "cmd_type": openapi.Schema(type=openapi.TYPE_STRING),
                "fieldindex": openapi.Schema(type=openapi.TYPE_INTEGER),
                "fieldname": openapi.Schema(type=openapi.TYPE_STRING),
                "fieldsize": openapi.Schema(type=openapi.TYPE_INTEGER),
                "value": openapi.Schema(type=openapi.TYPE_STRING),
                "minvalue": openapi.Schema(type=openapi.TYPE_STRING),
                "maxvalue": openapi.Schema(type=openapi.TYPE_STRING),
                "datatype": openapi.Schema(type=openapi.TYPE_INTEGER),
                "operation_in": openapi.Schema(type=openapi.TYPE_INTEGER),
                "operation_in_num": openapi.Schema(type=openapi.TYPE_INTEGER),
                "operation_out": openapi.Schema(type=openapi.TYPE_INTEGER),
                "operation_out_num": openapi.Schema(type=openapi.TYPE_INTEGER),
                "operabo_in": openapi.Schema(type=openapi.TYPE_INTEGER),
                "operabo_out": openapi.Schema(type=openapi.TYPE_INTEGER),
                "lua_script_in": openapi.Schema(type=openapi.TYPE_STRING),
                "lua_script_out": openapi.Schema(type=openapi.TYPE_STRING)
            }))
        })
    }
))
@api_view(['POST'])
def test(request):
    protocol_cmd = TableDevCmdNamePollView()
    cmd_fields = TableAllDevCmdDefineView()
    # print(request.data)
    protocol_cmd.request = request
    protocol_cmd.format_kwarg = None  # 设置 format_kwarg 属性
    
    cmd_fields.request = request
    cmd_fields.format_kwarg = None  # 设置 format_kwarg 属性

    cmds = request.data.get('cmds')
    for cmd in cmds.values():
        # 将指令的字段属性从字典中弹出
        fields = cmd.pop('fields')

        # 创建协议指令
        protocol_cmd_serializer = protocol_cmd.get_serializer(data=cmd)
        protocol_cmd_serializer.is_valid(raise_exception=True)
        cmd_explain = protocol_cmd_serializer.validated_data.get('cmd_explain')
        try:
            json.loads(cmd_explain)
        except json.JSONDecodeError:
            cmd_explain_dict = {
                'explain': cmd_explain,
                'version': "20230101"
            }
            cmd_explain = json.dumps(cmd_explain_dict)
        protocol_cmd_serializer.validated_data['cmd_explain'] = cmd_explain
        protocol_cmd.perform_create(protocol_cmd_serializer)
        # 创建指令
        for field in fields:
            cmd_fields_serializer = cmd_fields.get_serializer(data=field)
            cmd_fields_serializer.is_valid(raise_exception=True)
            cmd_fields.perform_create(cmd_fields_serializer)
    return Response(status=status.HTTP_201_CREATED)
